;******************************************************************************
; Tilemap Demo
;******************************************************************************
; Draw a tilemap to the screen using libperspective.
;
; This demo uses both banks of WRAM to draw and render a tilemap using a
; custom buffer copying routine and a modified sprite drawing macro. We do
; this so we can draw the tilemap once and simply copy it every frame without
; having to recalculate it.
;
; This example is probably the most complex. Readers who want to understand
; T_Draw_Tile will have to read the called function in libperspective to see
; what we replace by jumping into the function. T_Draw_Tile shows how
; libperspective can sometimes be hacked to do more without code duplication.
;
; (C) Kresna Susila - 2018
;******************************************************************************

;------------------------------------------------------------------------------
; Variables
;------------------------------------------------------------------------------
t_tile_x_counter        =       $6      ; 1 byte
t_tile_y_counter        =       $7      ; 1 byte
t_tile_x_loc            =       $8      ; 1 byte
t_tile_y_loc            =       $9      ; 1 byte
t_tile_address          =       $a      ; 2 bytes
t_sprite_x              =       $c      ; 1 byte
t_sprite_y              =       $d      ; 1 byte
t_sprite_address        =       $e      ; 2 bytes
t_sprite_size_x         =       $c8     ; 1 byte
t_sprite_size_y         =       $c9     ; 1 byte
t_tilemap_buffer        =       $b0     ; 26 bytes
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
; Constants
;------------------------------------------------------------------------------
T_TILEMAP_BUFFER_PTR    equ     $b0

T_BTN_RIGHT             equ     3
T_BTN_LEFT              equ     2
T_BTN_DOWN              equ     1
T_BTN_UP                equ     0
;------------------------------------------------------------------------------


Tilemap_Demo:
        ;----------------------------------------------------------------------
        ; Initialise and fill tilemap buffer
        ;----------------------------------------------------------------------
        ; We want this to go FAST
        clr1    ocr, 5

        ; Fill the tilemap buffer
        mov     #T_TILEMAP_BUFFER_PTR, 1        ; libperspective uses @R0
        mov     #<T_Tilemap, trl
        mov     #>T_Tilemap, trh
        mov     #0, c
        mov     #(T_Tilemap_End-T_Tilemap), b
.fill_buffer
        ld      c
        ldc
        st      @R1
        inc     1
        inc     c
        dbnz    b, .fill_buffer

        ; Fill the first bank of WRAM with tiles, using a modified macro
        ; to draw sprites into WRAM Bank 0.
        mov     #T_TILEMAP_BUFFER_PTR, 1        ; libperspective uses @R0
        ; Clear all these values
        xor     acc     ; acc = 0
        st      t_tile_x_counter
        st      t_tile_y_counter
        st      t_tile_x_loc
        st      t_tile_y_loc

        ; For each tile
.fill_tile
        ; Load the tile value
        mov     #<Tile_Table, trl
        mov     #>Tile_Table, trh

        ; Load and store the tile address
        ld      @R1
        rol
        ldc
        st      t_tile_address
        ld      @R1
        rol
        inc     acc
        ldc
        st      t_tile_address+1

        ; Calculate the tile coordinate
        ; t_tile_x_loc = t_tile_x_counter 8
        ld      t_tile_x_counter
        rol
        rol
        rol
        st      t_tile_x_loc
        ; t_tile_y_loc = t_tile_y_loc * 8
        ld      t_tile_y_counter
        rol
        rol
        rol
        st      t_tile_y_loc

        ; Draw the tile into our framebuffer
        T_Draw_Tile t_tile_address, t_tile_x_loc, t_tile_y_loc

        ; libperspective disables RC after each call, so enable it again
        clr1    ocr, 5

        inc     t_tile_x_counter
        ; If t_tile_x_counter == 6 (End of screen)
        ld      t_tile_x_counter
        sub     #6
        bnz     .skip_line_adjust

        ; Increase t_tile_y_counter
        inc     t_tile_y_counter
        ; Reset X Counter
        mov     #0, t_tile_x_counter

.skip_line_adjust

        ; Increment the ram pointer
        inc     1
        ; If the ram pointer is not at the end
        ld      1
        sub     #(T_Tilemap_End-T_Tilemap+T_TILEMAP_BUFFER_PTR)
        bnz     .fill_tile
        ;----------------------------------------------------------------------

        ; Initialise some sprite variables
        mov     #16, t_sprite_x
        mov     #16, t_sprite_y
        mov     #<heart_mask, acc
        st      trl
        st      t_sprite_address
        mov     #>heart_mask, acc
        st      trh
        st      t_sprite_address+1
        xor     acc
        ldc
        st      t_sprite_size_x
        mov     #1, acc
        ldc
        st      t_sprite_size_y
.demo_loop
        ;----------------------------------------------------------------------
        ; Demo Loop Code
        ;----------------------------------------------------------------------
        ; If player presses left
        ld      p3
        bp      acc, T_BTN_LEFT, .not_left
        ; If t_sprite_x != 0
        ld      t_sprite_x
        bz      .not_left
        dec     t_sprite_x
        br      .not_right
.not_left
        ; If player presses right
        ld      p3
        bp      acc, T_BTN_RIGHT, .not_right
        ; if t_sprite_x + t_sprite_size_x != 48
        ld      t_sprite_x
        add     t_sprite_size_x
        sub     #48
        bz      .not_right
        inc     t_sprite_x
.not_right
        ; If player presses up
        ld      p3
        bp      acc, T_BTN_UP, .not_up
        ; If t_sprite_y != 0
        ld      t_sprite_y
        bz      .not_up
        dec     t_sprite_y
        br      .not_down
.not_up
        ; If player presses down
        ld      p3
        bp      acc, T_BTN_DOWN, .not_down
        ; If t_sprite_y + t_sprite_size_y != 32
        ld      t_sprite_y
        add     t_sprite_size_y
        sub     #32
        bz      .not_down
        inc     t_sprite_y
.not_down

        ;----------------------------------------------------------------------
        ; Scene Draw
        ;----------------------------------------------------------------------
        ; Copy the tile buffer
        T_Copy_Buffer
        ; Draw the sprite
        P_Draw_Sprite_Mask t_sprite_address, t_sprite_x, t_sprite_y
        P_Blit_Screen
        ; Generic input code to move between demos
        call    Get_Input
        mov     #Button_A, acc
        call    Check_Button_Pressed
        bz      .not_a
        ret
.not_a
        jmp     .demo_loop


; Tilemap data
T_Tilemap:
        .byte   1, 2, 2, 2, 1, 1
        .byte   1, 0, 5, 0, 1, 1
        .byte   3, 3, 3, 3, 3, 3
        .byte   4, 4, 4, 4, 4, 4
T_Tilemap_End:

%macro  T_Copy_Buffer
        ;----------------------------------------------------------------------
        ; Give our nice function a pretty macro alias
        ;----------------------------------------------------------------------
        call    _T_Copy_Buffer
%end

_T_Copy_Buffer:
        ;----------------------------------------------------------------------
        ; A custom function to copy from WRAM0 to WRAM1
        ;----------------------------------------------------------------------
        ; Gotta go fast
        clr1    ocr, 5
        ; We'll use libperspective's constants
        mov     #P_WRAM_ADDR, vrmad1
        mov     #0, vrmad2
        mov     #24, c
        mov     #%00000000, vsel
        ; Use a relatively unrolled loop to do a fast copy
.copy
        ; 1 copy
        ld      vtrbf
        mov     #1, vrmad2
        st      vtrbf
        mov     #0, vrmad2
        inc     vrmad1

        ld      vtrbf
        mov     #1, vrmad2
        st      vtrbf
        mov     #0, vrmad2
        inc     vrmad1
        ld      vtrbf
        mov     #1, vrmad2
        st      vtrbf
        mov     #0, vrmad2
        inc     vrmad1
        ld      vtrbf
        mov     #1, vrmad2
        st      vtrbf
        mov     #0, vrmad2
        inc     vrmad1
        ld      vtrbf
        mov     #1, vrmad2
        st      vtrbf
        mov     #0, vrmad2
        inc     vrmad1
        ld      vtrbf
        mov     #1, vrmad2
        st      vtrbf
        mov     #0, vrmad2
        inc     vrmad1
        ld      vtrbf
        mov     #1, vrmad2
        st      vtrbf
        mov     #0, vrmad2
        inc     vrmad1
        ld      vtrbf
        mov     #1, vrmad2
        st      vtrbf
        mov     #0, vrmad2
        inc     vrmad1
        dbnz    c, .copy
        set1    ocr, 5
        ret

%macro  T_Draw_Tile %spr_addr, %spr_x, %spr_y
        ;----------------------------------------------------------------------
        ; Draw a tile to bank 0 of WRAM with no mask using stored RAM values
        ;----------------------------------------------------------------------
        ; This macro is a modification of P_Draw_Sprite. It preloads the
        ; registers required by _P_Draw_Sprite_No_Mask, then prepares the
        ; WRAM bank registers before jumping past the init code of
        ; _P_Draw_Sprite_No_Mask because we're gnarly.
        ;
        ; Accepts:      Sprite Address (16bit Constant)
        ;               Sprite X Coordinate (8bit Constant)
        ;               Sprite Y Coordinate (8bit Constant)
        ; Destroys:     ACC, B, C, TRL, TRH, VRMAD1, VRMAD2, VSEL, PSW
        ;----------------------------------------------------------------------
        ; Preload RAM
        ld      %spr_x
        st      p_spr_x
        ld      %spr_y
        st      p_spr_y
        ld      %spr_addr
        st      trl
        ld      %spr_addr+1
        st      trh

        ; A recreation with modifications of _P_Draw_Sprite_No_Mask's init code
        clr1    ocr, 5                          ; 2 bytes
        mov     #0, vrmad2                      ; 3 bytes // This has changed
        mov     #%00010000, vsel                ; 3 bytes
        call    _P_Draw_Sprite_No_Mask+8        ; (2+3+3 bytes)
%end
